/*
  EK_ADS7828 Example
  Based on Adafruit_ADS7830 (BSD License)

  Original code Copyright (c) Adafruit Industries
  Modified code Copyright (c) 2026 Electrokit Sweden AB

  This example is released under the BSD 2-Clause license. 
  Redistribution and use in source and binary forms, with 
  or without modification, are permitted as long as the original 
  copyright notice and this permission notice remain in place.
*/

#include <Wire.h>
#include "EK_ADS7828.h"

EK_ADS7828 adc;

const bool USE_INTERNAL_REF    = false;                                   // true = int vref, false = ext vref | Do NOT connect VREF to anything when set to true!
const float VREF_EXTERNAL      = 3.3;                                     // ext vref max voltage (only used for calculated voltage)
const float VREF               = USE_INTERNAL_REF ? 2.5 : VREF_EXTERNAL;  // display correct voltage in serial monitor, internal (2.5V) or external
const bool READ_SINGLE_CHANNEL = false;                                   // true = read only one channel
const uint8_t SINGLE_CHANNEL   = 0;                                       // which channel to read when READ_SINGLE_CHANNEL = true
const bool PRINT_VOLTAGE       = true;                                    // print volts as an extra row
const uint8_t I2C_ADDRESS      = 0x48;                                    // set device I2C address. 0x48 is default (0:GND,1:GND). possible options: 0x49 (0:VCC,1:GND) / 0x4A (0:GND,1:VCC) / 0x4B (0:VCC,1:VCC)
const uint32_t I2C_CLOCK_HZ    = 400000;                                  // Set SCL clock frequency in Hz. 100000 (100kHz), 400000 (400kHz), 1000000 (1MHz)

static int16_t readSingleSettled(uint8_t ch, ek7828PowerDownSelection pd) {
  (void)adc.readADCsingle(ch, pd);
  delayMicroseconds(50);
  return adc.readADCsingle(ch, pd);
}

void setup() {
  Serial.begin(115200);
  while (!Serial) { }

  Wire.begin();
  Wire.setWireTimeout(25000, true);

  adc.setI2CClock(I2C_CLOCK_HZ);

  if (!adc.begin(I2C_ADDRESS)) {
    Serial.println("ADS7828 not found");
    while (1) { }
  }

  Serial.println();
  Serial.println("ADS7828 Single-Ended Demo");
  Serial.print("Reference: ");
  Serial.println(VREF, 3);

  Serial.print("Channels: ");
  if (READ_SINGLE_CHANNEL) {
    Serial.print("single (CH");
    Serial.print(SINGLE_CHANNEL);
    Serial.println(")");
  } else {
    Serial.println("all 8 channels (CH0..CH7)");
  }

  Serial.println();
  printHeader();
}

void loop() {
  ek7828PowerDownSelection pd =
      USE_INTERNAL_REF ? INTERNAL_REF_ON_ADC_ON : INTERNAL_REF_OFF_ADC_ON;

  uint8_t firstCh, lastCh;
  if (READ_SINGLE_CHANNEL) {
    firstCh = SINGLE_CHANNEL;
    lastCh  = SINGLE_CHANNEL;
  } else {
    firstCh = 0;
    lastCh  = 7;
  }

  const uint8_t maxChannels = 8;
  int16_t values[maxChannels];
  float voltages[maxChannels];

  for (uint8_t ch = firstCh; ch <= lastCh; ch++) {
    int16_t raw;
    raw = readSingleSettled(ch, pd);
    values[ch] = raw;
    if (raw >= 0) {
      voltages[ch] = (raw / 4095.0f) * VREF;
    } else {
      voltages[ch] = NAN;
    }
  }

  printValuesRow(values, firstCh, lastCh);
  if (PRINT_VOLTAGE) {
    printVoltageRow(voltages, firstCh, lastCh);
  }

  delay(500);
}

void printHeader() {
  uint8_t firstCh, lastCh;
  if (READ_SINGLE_CHANNEL) {
    firstCh = SINGLE_CHANNEL;
    lastCh  = SINGLE_CHANNEL;
  } else {
    firstCh = 0;
    lastCh  = 7;
  }

  Serial.print(" ");
  for (uint8_t ch = firstCh; ch <= lastCh; ch++) {
    char buf[16];
    snprintf(buf, sizeof(buf), "CH%u:  ", ch);
    Serial.print(buf);

    if (ch < lastCh) {
      Serial.print("|  ");
    }
  }
  Serial.println();

}

void printValuesRow(int16_t *values, uint8_t firstCh, uint8_t lastCh) {

  for (uint8_t ch = firstCh; ch <= lastCh; ch++) {
    char buf[16];
    if (values[ch] >= 0) {
      snprintf(buf, sizeof(buf), "%6d", values[ch]);
    } else {
      // error
      snprintf(buf, sizeof(buf), "  ERR");
    }
    Serial.print(buf);

    if (ch < lastCh) {
      Serial.print(" | ");
    }
  }
  Serial.println();
}

void printVoltageRow(float *voltages, uint8_t firstCh, uint8_t lastCh) {
  for (uint8_t ch = firstCh; ch <= lastCh; ch++) {
    char buf[16];

    if (!isnan(voltages[ch])) {

      dtostrf(voltages[ch], 6, 3, buf);
    } else {
      snprintf(buf, sizeof(buf), "  ---");
    }

    Serial.print(buf);

    if (ch < lastCh) {
      Serial.print(" | ");
    }
  }
  Serial.println();
  Serial.println();
}
