/*
  EK_ADS7828 Example
  Based on Adafruit_ADS7830 (BSD License)

  Original code Copyright (c) Adafruit Industries
  Modified code Copyright (c) 2026 Electrokit Sweden AB

  This example is released under the BSD 2-Clause license.
  Redistribution and use in source and binary forms, with
  or without modification, are permitted as long as the original
  copyright notice and this permission notice remain in place.
*/

#include <Wire.h>
#include "EK_ADS7828.h"

EK_ADS7828 adc;

const bool USE_INTERNAL_REF    = false;                                   // true = int vref, false = ext vref | Do NOT connect VREF to anything when set to true!
const float VREF_EXTERNAL      = 3.3;                                     // ext vref max voltage (only used for calculated voltage)
const float VREF               = USE_INTERNAL_REF ? 2.5 : VREF_EXTERNAL;  // display correct voltage in serial monitor, internal (2.5V) or external
const bool PRINT_VOLTAGE       = true;                                    // print volts as an extra row
const uint8_t I2C_ADDRESS      = 0x48;                                    // set device I2C address. 0x48 is default (0:GND,1:GND). possible options: 0x49 (0:VCC,1:GND) / 0x4A (0:GND,1:VCC) / 0x4B (0:VCC,1:VCC)
const uint32_t I2C_CLOCK_HZ    = 400000;                                  // Set SCL clock frequency in Hz. 100000 (100kHz), 400000 (400kHz), 1000000 (1MHz)

static inline float rawToVolts(int16_t raw) {
  return (raw / 4095.0f) * VREF;
}

static int16_t readDiffSettled(ek7828ChannelSelectionControl sel, ek7828PowerDownSelection pd) {
  (void)adc.readADCdifferential(sel, pd);
  delayMicroseconds(50);
  return adc.readADCdifferential(sel, pd);
}

static int16_t readSingleSettled(uint8_t ch, ek7828PowerDownSelection pd) {
  (void)adc.readADCsingle(ch, pd);
  delayMicroseconds(50);
  return adc.readADCsingle(ch, pd);
}

void setup() {
  Serial.begin(115200);
  while (!Serial) {}

  Wire.begin();
  Wire.setWireTimeout(25000, true);

  adc.setI2CClock(I2C_CLOCK_HZ);

  if (!adc.begin(I2C_ADDRESS)) {
    Serial.println("ADS7828 not found");
    while (1) { }
  }

  Serial.println();
  Serial.println("ADS7828 Pseudo-Differential Demo");
  Serial.print("Reference: ");
  Serial.println(VREF, 3);
  Serial.println();

  Serial.println("  CH0+  CH1-  Diff |   CH2+  CH3-  Diff |   CH4+  CH5-  Diff |   CH6+  CH7-  Diff");
  Serial.println("------------------------------------------------------------------------------");
}

void loop() {
  ek7828PowerDownSelection pd =
      USE_INTERNAL_REF ? INTERNAL_REF_ON_ADC_ON : INTERNAL_REF_OFF_ADC_ON;

  int16_t plusRaw[4], minusRaw[4], diffRaw[4];
  float plusV[4], minusV[4], diffV[4];

  // pairs: (0,1), (2,3), (4,5), (6,7)
  const uint8_t plusCh[4]  = {0, 2, 4, 6};
  const uint8_t minusCh[4] = {1, 3, 5, 7};
  const ek7828ChannelSelectionControl diffSel[4] = {
    DIFF_CH0_CH1, DIFF_CH2_CH3, DIFF_CH4_CH5, DIFF_CH6_CH7
  };

for (uint8_t i = 0; i < 4; i++) {
  plusRaw[i]  = readSingleSettled(plusCh[i], pd);
  minusRaw[i] = readSingleSettled(minusCh[i], pd);
  diffRaw[i]  = readDiffSettled(diffSel[i], pd);

  plusV[i]  = (plusRaw[i]  >= 0) ? rawToVolts(plusRaw[i])  : NAN;
  minusV[i] = (minusRaw[i] >= 0) ? rawToVolts(minusRaw[i]) : NAN;
  diffV[i]  = (diffRaw[i]  >= 0) ? rawToVolts(diffRaw[i])  : NAN;
}

printDiffRawRow(plusRaw, minusRaw, diffRaw);
if (PRINT_VOLTAGE) printDiffVoltRow(plusV, minusV, diffV);


  Serial.println();
  delay(500);
}

void printDiffRawRow(int16_t *plusRaw, int16_t *minusRaw, int16_t *diffRaw) {

  for (uint8_t i = 0; i < 4; i++) {
    char b[16];

    if (plusRaw[i] >= 0)  snprintf(b, sizeof(b), "%6d", plusRaw[i]);
    else                  snprintf(b, sizeof(b), "  ERR");
    Serial.print(b);

    if (minusRaw[i] >= 0) snprintf(b, sizeof(b), "%6d", minusRaw[i]);
    else                  snprintf(b, sizeof(b), "  ERR");
    Serial.print(b);

    if (diffRaw[i] >= 0)  snprintf(b, sizeof(b), "%6d", diffRaw[i]);
    else                  snprintf(b, sizeof(b), "  ERR");
    Serial.print(b);

    if (i < 3) Serial.print(" | ");
  }
  Serial.println();
}

void printDiffVoltRow(float *plusV, float *minusV, float *diffV) {
  for (uint8_t i = 0; i < 4; i++) {
    char b[16];

    if (!isnan(plusV[i]))  dtostrf(plusV[i],  6, 3, b);
    else                   snprintf(b, sizeof(b), "  ---");
    Serial.print(b);

    if (!isnan(minusV[i])) dtostrf(minusV[i], 6, 3, b);
    else                   snprintf(b, sizeof(b), "  ---");
    Serial.print(b);

    if (!isnan(diffV[i]))  dtostrf(diffV[i],  6, 3, b);
    else                   snprintf(b, sizeof(b), "  ---");
    Serial.print(b);

    if (i < 3) Serial.print(" | ");
  }
  Serial.println();
  Serial.println();
}