/*
  EK_ADS7828 - ADS7828 I2C ADC library
  Derived from "Adafruit_ADS7830" by Adafruit Industries

  Original work Copyright (c) Adafruit Industries
  Modified work Copyright (c) 2026 Electrokit Sweden AB

  This library is free software; you can redistribute it and/or modify it
  under the terms of the BSD license. Redistribution must retain the above
  copyright notices and this permission notice.
*/

#include "EK_ADS7828.h"
#include <Wire.h>

// Small helper: request I2C speed on the active Wire instance.
// No clamping - if the platform/bus can't handle it, the user can lower it.
static void _ek7828_apply_i2c_clock(TwoWire *wire, uint32_t hz) {
  if (!wire) return;
  wire->setClock(hz);
}

// ---------- Construction / configuration ----------

EK_ADS7828::EK_ADS7828()
  : i2c_addr(EK_ADS7828_DEFAULT_ADDR),
    i2c_clock(EK_ADS7828_DEFAULT_I2C_CLOCK),
    wire(&Wire),
    i2c_dev(nullptr) {}

void EK_ADS7828::setI2CClock(uint32_t hz) {
  i2c_clock = hz;
  _ek7828_apply_i2c_clock(wire, i2c_clock);
}

uint8_t EK_ADS7828::getAddress() const {
  return i2c_addr;
}

bool EK_ADS7828::begin(TwoWire *theWire) {
  return begin(i2c_addr, theWire);
}

bool EK_ADS7828::begin(uint8_t addr, TwoWire *theWire) {
  i2c_addr = addr;
  wire = theWire;

  if (i2c_dev) {
    delete i2c_dev;
    i2c_dev = nullptr;
  }

  i2c_dev = new Adafruit_I2CDevice(i2c_addr, wire);

  // Apply desired clock BEFORE begin (good practice)
  _ek7828_apply_i2c_clock(wire, i2c_clock);

  bool ok = i2c_dev->begin();

  // Important: some versions of Adafruit_BusIO may reset Wire clock during begin().
  // Re-apply our requested clock AFTER begin.
  _ek7828_apply_i2c_clock(wire, i2c_clock);

  return ok;
}

// ---------- Command helpers ----------

static uint8_t _ek7828_single_command(uint8_t ch, ek7828PowerDownSelection pd) {
  uint8_t sel = 0;

  // Single-ended channel selection order:
  // CH0,2,4,6 then CH1,3,5,7
  if ((ch % 2) == 0) {
    sel = (uint8_t)(SINGLE_CH0 + (ch / 2));
  } else {
    sel = (uint8_t)(SINGLE_CH1 + ((ch - 1) / 2));
  }

  return (uint8_t)((sel << 4) | (pd << 2));
}

// FIXED: odd channels map to reverse differential pairs.
// Even: 0->CH0-CH1, 2->CH2-CH3, 4->CH4-CH5, 6->CH6-CH7
// Odd:  1->CH1-CH0, 3->CH3-CH2, 5->CH5-CH4, 7->CH7-CH6
static uint8_t _ek7828_diff_command_from_channel(uint8_t ch, ek7828PowerDownSelection pd) {
  uint8_t sel = 0;

  if ((ch % 2) == 0) {
    sel = (uint8_t)(DIFF_CH0_CH1 + (ch / 2));
  } else {
    sel = (uint8_t)(DIFF_CH1_CH0 + ((ch - 1) / 2));
  }

  return (uint8_t)((sel << 4) | (pd << 2));
}

static uint8_t _ek7828_command_from_sel(ek7828ChannelSelectionControl sel,
                                       ek7828PowerDownSelection pd) {
  return (uint8_t)(((uint8_t)sel << 4) | ((uint8_t)pd << 2));
}

static int16_t _ek7828_read12(Adafruit_I2CDevice *dev, uint8_t cmd) {
  if (!dev) return -1;

  uint8_t data[2];
  if (!dev->write_then_read(&cmd, 1, data, 2)) return -1;

  // 12-bit straight binary
  return (int16_t)(((data[0] & 0x0F) << 8) | data[1]);
}

// ---------- Public read functions ----------

int16_t EK_ADS7828::readADCsingle(uint8_t ch, ek7828PowerDownSelection pd) {
  if (ch > 7) return -1;

  uint8_t cmd = _ek7828_single_command(ch, pd);
  return _ek7828_read12(i2c_dev, cmd);
}

int16_t EK_ADS7828::readADCdifferential(uint8_t ch, ek7828PowerDownSelection pd) {
  if (ch > 7) return -1;

  uint8_t cmd = _ek7828_diff_command_from_channel(ch, pd);
  return _ek7828_read12(i2c_dev, cmd);
}

int16_t EK_ADS7828::readADCdifferential(ek7828ChannelSelectionControl sel,
                                       ek7828PowerDownSelection pd) {
  if ((uint8_t)sel > 0x07) return -1; // only DIFF selections

  uint8_t cmd = _ek7828_command_from_sel(sel, pd);
  return _ek7828_read12(i2c_dev, cmd);
}