/*
  LIS3DH 3-axis accelerometer Example

  Code Copyright (c) 2026 Electrokit Sweden AB

  Uses Sparkfun LIS3DH library (https://github.com/sparkfun/SparkFun_LIS3DH_Arduino_Library)

  This example is released under the BSD 2-Clause license.
  Redistribution and use in source and binary forms, with
  or without modification, are permitted as long as the original
  copyright notice and this permission notice remain in place.
*/

#include "SparkFunLIS3DH.h"
#include "Wire.h"
#include "SPI.h"
#include <math.h>

const uint8_t PLOT_MODE = 0;        // 0 = ax, ay, az (g)
                                    // 1 = mag, ax, ay, az (g)
                                    // 2 = ax, ay, az, tDelta
                                    // 3 = ax, ay, az, adc1, adc2, adc3 (10-bit)
                                    // 4 = tDelta, adc1, adc2 (ADC3 is temperature sensor)

const uint16_t SAMPLE_HZ = 100;    // 0,1,10,25,50,100,200,400,1600,5000
const uint8_t ACCEL_RANGE_G = 2;   // 2,4,8,16

const bool ENABLE_TEMP = false;    // Uses ADC3 channel
const bool ENABLE_ADC  = true;     // ADC1/2 always usable; ADC3 only if temp disabled

const uint8_t LIS3DH_ADDR = 0x19;  // 0x18 (default) or 0x19
const uint16_t LOOP_DELAY_MS = 10; // 10 ms ~100 Hz

LIS3DH myIMU(I2C_MODE, LIS3DH_ADDR);

static float accelMagnitude(float ax, float ay, float az) {
  return sqrtf(ax * ax + ay * ay + az * az);
}

static int t0 = 0;          // Temperature baseline
static bool t0_set = false;

void setup() {
  Serial.begin(115200);
  delay(200);

  myIMU.settings.accelSampleRate = SAMPLE_HZ;
  myIMU.settings.accelRange = ACCEL_RANGE_G;

  myIMU.settings.xAccelEnabled = 1;
  myIMU.settings.yAccelEnabled = 1;
  myIMU.settings.zAccelEnabled = 1;

  myIMU.settings.tempEnabled = ENABLE_TEMP ? 1 : 0;
  myIMU.settings.adcEnabled  = ENABLE_ADC  ? 1 : 0;

  myIMU.begin();

}

void loop() {
  // Accel (g)
  float ax = myIMU.readFloatAccelX();
  float ay = myIMU.readFloatAccelY();
  float az = myIMU.readFloatAccelZ();

  int adc1 = 0, adc2 = 0, adc3 = 0;  // range 1.2V centered around Vmid

  if (ENABLE_ADC) {
    adc1 = myIMU.read10bitADC1();
    adc2 = myIMU.read10bitADC2();

    // Only read ADC3 as external channel if temp is NOT enabled
    if (!ENABLE_TEMP) {
      adc3 = myIMU.read10bitADC3();
    }
  }

  int tRaw = 0;
  int tDelta = 0;
  if (ENABLE_TEMP) {
    tRaw = myIMU.read10bitADC3();
    if (!t0_set) {
      t0 = tRaw;     // baseline at startup
      t0_set = true;
    }
    //tDelta = tRaw - t0; // if tDelta is inverted use the line below.
    tDelta = t0 - tRaw;   // if tDelta is inverted use the line above.
  }

  switch (PLOT_MODE) {
    case 0:
      Serial.print("ax:"); Serial.print(ax, 4);
      Serial.print(" ay:"); Serial.print(ay, 4);
      Serial.print(" az:"); Serial.println(az, 4);
      break;

    case 1: {
      float mag = accelMagnitude(ax, ay, az);
      Serial.print("mag:"); Serial.print(mag, 4);
      Serial.print(" ax:"); Serial.print(ax, 4);
      Serial.print(" ay:"); Serial.print(ay, 4);
      Serial.print(" az:"); Serial.println(az, 4);
      break;
    }

    case 2:
      Serial.print("ax:"); Serial.print(ax, 4);
      Serial.print(" ay:"); Serial.print(ay, 4);
      Serial.print(" az:"); Serial.print(az, 4);
      Serial.print(" tDelta:"); Serial.println(tDelta);
      break;

    case 3:
      Serial.print("ax:"); Serial.print(ax, 4);
      Serial.print(" ay:"); Serial.print(ay, 4);
      Serial.print(" az:"); Serial.print(az, 4);
      Serial.print(" adc1:"); Serial.print(adc1);
      Serial.print(" adc2:"); Serial.print(adc2);
      Serial.print(" adc3:"); Serial.println(adc3);
      break;

    case 4:
      Serial.print("tDelta:"); Serial.print(tDelta);
      Serial.print(" adc1:"); Serial.print(adc1);
      Serial.print(" adc2:"); Serial.println(adc2);
      break;

    default:
      Serial.print("ax:"); Serial.print(ax, 4);
      Serial.print(" ay:"); Serial.print(ay, 4);
      Serial.print(" az:"); Serial.println(az, 4);
      break;
  }

  delay(LOOP_DELAY_MS);
}
